import enum

from matrx.agents.agent_utils.navigator import Navigator, AStarPlanner
from matrx.agents.agent_utils.state_tracker import StateTracker
from matrx.agents import AgentBrain
from moving_out.agents.non_faulty_agent import world_width, world_height
from matrx.messages import Message
from moving_out.actions.custom_actions import CarryObject, LowerObject, dropzone, safezone
from matrx.actions.object_actions import GrabObject, DropObject
from matrx.objects import EnvObject
import time

safeloc = [12,5]
tutorial_build_boxes = [
    {"location": (2,3), "weight": "light", "owner": "human"},
    {"location": (4,9), "weight": "light", "owner": "robot"},
    {"location": (17,5), "weight": "heavy", "owner": "human"},
    {"location": (10,7), "weight": "heavy", "owner": "human"}
    ]


class Todo(enum.Enum):
    # add todos
    WAIT = 0,
    WAIT_FOR_HUMAN = 1,
    WAIT_FOR_DROP = 2,
    LOWER_BOX = 3,
    CHECK_FOR_PICKUP = 4,
    GO_TO_HUMAN = 5,
    STOP_WAITING = 6


class TutorialAgent(AgentBrain):
    """ An artificial agent whose behaviour can be programmed to be, for example, (semi-)autonomous.

    For more extensive documentation on the functions below, see:
    http://docs.matrx-software.com/en/master/sections/_generated_autodoc/matrx.agents.agent_brain.AgentBrain.html#matrx.agents.agent_brain.AgentBrain
    """

    def __init__(self, max_carry_objects=1,
                 grab_range=0, **kwargs):
        """ Creates an agent brain to move along a set of waypoints.
        """
        super().__init__(**kwargs)
        self.waypoints = [] if 'waypoints' not in kwargs else kwargs['waypoints']
        self.__max_carry_objects = max_carry_objects
        self.__grab_range = grab_range
        self.todo = Todo.WAIT
        self.box = None
        self.next_todo = None
        self.carrying_same_box = False
        self.humanloc = None

    def initialize(self):
        # initialize state tracker
        self.state_tracker = StateTracker(agent_id=self.agent_id)
        self.navigator = Navigator(agent_id=self.agent_id, action_set=self.action_set,
                                   algorithm=Navigator.A_STAR_ALGORITHM)
        self.navigator.add_waypoints(self.waypoints, is_circular=False)

    def filter_observations(self, state):
        """ Filters the world state before deciding on an action. """
        return state

    def decide_on_action(self, state):
        """ Contains the decision logic of the agent. """
        action = None
        action_kwargs = {"action_duration": 20}
        # Set grab range
        action_kwargs['grab_range'] = self.__grab_range
        # Set max amount of objects
        action_kwargs['max_objects'] = self.__max_carry_objects

        while True:
            # set image back to not having exclamation mark
            if self.todo == Todo.STOP_WAITING:
                self.agent_properties["img_name"] = "robot.png"

            # get all objects
            objects = state.get_objects_in_area((0, 0), world_width, world_height,
                                                (world_width, world_height))
            # filter out objects to keep all ghostboxes
            ghostboxes = [x for x in objects if x['name'] == 'ghostbox']
            # filter out objects to keep all boxes
            boxes = [x for x in objects if x['name'] == 'box']
            # check messages
            for message in list(self.received_messages):
                if message.content == 'LOWER-BOX':
                    if self.todo == Todo.WAIT_FOR_DROP:
                        self.todo = Todo.LOWER_BOX
                    elif self.todo == Todo.WAIT_FOR_HUMAN:
                        self.todo = Todo.STOP_WAITING
                elif message.content == 'HELP-HEAVY':
                    if self.todo == Todo.CHECK_FOR_PICKUP:
                        human = state.get_agents_with_property({"name": "human"})[0]
                        self.humanloc = human.get('location')  # to get the location where the human was at that moment
                        self.todo = Todo.GO_TO_HUMAN
                self.received_messages.remove(message)
            if len(tutorial_build_boxes) - len(ghostboxes) == 0:
                self.todo = Todo.WAIT
            if len(tutorial_build_boxes) - len(ghostboxes) == 1:
                self.todo = Todo.WAIT
            # human has to indicate pickup
            if len(tutorial_build_boxes) - len(ghostboxes) == 2:
                if self.todo == Todo.WAIT:
                    self.todo = Todo.CHECK_FOR_PICKUP
            # robot indicates pickup
            if len(tutorial_build_boxes) - len(ghostboxes) == 3:
                if self.todo == Todo.WAIT:
                    props = ghostboxes[0]
                    obj = state.get_closest_with_property(
                        props={'weight': props['weight'], 'owner': props['owner'], 'is_movable': True})
                    if obj:
                        obj[0]['destination'] = props['location']
                        self.box = obj[0]
                        self.box['destination'] = props['location']
                        self.navigator.reset_full()  # for some reasons it's looping if we don't reset the waypoints
                        self.navigator.add_waypoint(tuple(self.box["location"]))
                        self.state_tracker.update(state)
                        action = self.navigator.get_move_action(self.state_tracker)
                        if action != None:
                            return action, action_kwargs
                        self.todo = Todo.WAIT_FOR_HUMAN
                elif self.todo == Todo.WAIT_FOR_HUMAN:
                    self.agent_properties["img_name"] = "robot_help.png"
                    # get agents
                    robot = state.get_agents_with_property({"name": "robot"})[0]
                    human = state.get_agents_with_property({"name": "human"})[0]
                    # give id of box
                    action_kwargs['object_id'] = self.box['obj_id']
                    if robot['location'] == human['location']:
                        # set no waypoint cause human will decide direction
                        # set todo to wait for drop
                        self.todo = Todo.WAIT_FOR_DROP
            if self.todo == Todo.LOWER_BOX:
                # give destination with the function
                action_kwargs['destination'] = self.box['destination']
                # set todo to wait
                self.todo = Todo.WAIT
                # lower the box it is carrying
                return LowerObject.__name__, action_kwargs
            if self.todo == Todo.GO_TO_HUMAN:
                human = state.get_agents_with_property({"name": "human"})[0]
                if human.get('location') == self.humanloc:
                    # set waypoint to human location
                    self.navigator.reset_full()  # for some reasons it's looping if we don't reset the waypoints
                    self.navigator.add_waypoint(self.humanloc)
                    self.state_tracker.update(state)
                    action = self.navigator.get_move_action(self.state_tracker)
                    # get the box
                    closest_heavy_box = state.get_closest_with_property(
                        props={"weight": "heavy", "location": human.get("location")})
                    self.box = closest_heavy_box[0] if closest_heavy_box else None
                    # give id of box
                    action_kwargs['object_id'] = self.box['obj_id']
                    self.box['destination'] = ghostboxes[0]['location']
                    # wait until robot arrives
                    if action != None:
                        return action, action_kwargs
                    robot = state.get_agents_with_property({"name": "robot"})[0]
                    human = state.get_agents_with_property({"name": "human"})[0]
                    if robot['location'] == human['location']:
                        # set no waypoint cause human will decide direction
                        # set todo to wait for drop
                        self.todo = Todo.WAIT_FOR_DROP
                        return CarryObject.__name__, action_kwargs
                else:
                    self.navigator.reset_full()
            return action, action_kwargs

    def _set_messages(self, messages=None):
        # make sure we save the entire message and not only the content
        for mssg in messages:
            received_message = mssg
            self.received_messages.append(received_message)